<?php
/**
 * AEO WP Plugin Dashboard - Ultra-Safe Minimal Version
 * 
 * @package AEOWPPlugin
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// ULTRA-SAFE: Initialize all variables with defaults first
$has_results = false;
$scan_score = 0;
$scan_status = 'Unknown';
$scan_url = '';
$scan_timestamp = 0;
$issues_count = 0;

// Try to load summary data safely
try {
	$last_scan_summary = get_option( 'aeowpplugin_last_scan_summary', array() );
	
	if ( is_array( $last_scan_summary ) && ! empty( $last_scan_summary ) ) {
		$has_results = true;
		$scan_score = isset( $last_scan_summary['score'] ) ? intval( $last_scan_summary['score'] ) : 0;
		$scan_status = isset( $last_scan_summary['status'] ) ? sanitize_text_field( $last_scan_summary['status'] ) : 'Unknown';
		$scan_url = isset( $last_scan_summary['url'] ) ? esc_url( $last_scan_summary['url'] ) : '';
		$scan_timestamp = isset( $last_scan_summary['timestamp'] ) ? intval( $last_scan_summary['timestamp'] ) : 0;
		$issues_count = isset( $last_scan_summary['issues_count'] ) ? intval( $last_scan_summary['issues_count'] ) : 0;
	}
} catch ( Exception $e ) {
	// Silently fail - use defaults
	error_log( 'AEO WP Plugin: Error loading summary - ' . $e->getMessage() );
}

// Simple defaults - no class dependencies
$plan = 'free';
$plan_name = 'Free';
$quota_used = 0;
$quota_limit = 5;
$quota_remaining = 5;
$usage_percentage = 0;

// Try to get real data if classes are available
if ( class_exists( 'AEOWPPlugin_License_Manager' ) ) {
	try {
		$license_data = AEOWPPlugin_License_Manager::get_license_data();
		$plan = isset( $license_data['plan'] ) ? $license_data['plan'] : 'free';
		
		if ( function_exists( 'aeowpplugin_get_plan_name' ) ) {
			$plan_name = aeowpplugin_get_plan_name( $plan );
		} else {
			$plan_name = ucfirst( $plan );
		}
	} catch ( Exception $e ) {
		// Silently fail and use defaults
	}
}

if ( class_exists( 'AEOWPPlugin_Quota_Manager' ) ) {
	try {
		$quota = AEOWPPlugin_Quota_Manager::get_quota_status();
		$quota_used = isset( $quota['used'] ) ? $quota['used'] : 0;
		$quota_limit = isset( $quota['limit'] ) ? $quota['limit'] : 5;
		$quota_remaining = isset( $quota['remaining'] ) ? $quota['remaining'] : 5;
		$usage_percentage = AEOWPPlugin_Quota_Manager::get_usage_percentage();
	} catch ( Exception $e ) {
		// Silently fail and use defaults
	}
}

// Get plan badge class
$plan_badge_class = 'badge-' . $plan;
if ( function_exists( 'aeowpplugin_get_plan_badge_class' ) ) {
	try {
		$plan_badge_class = aeowpplugin_get_plan_badge_class( $plan );
	} catch ( Exception $e ) {
		// Use default
	}
}

// Get upgrade URL
$upgrade_url = 'https://aeowpplugin.com/pricing';
if ( function_exists( 'aeowpplugin_get_upgrade_url' ) ) {
	try {
		$upgrade_url = aeowpplugin_get_upgrade_url();
	} catch ( Exception $e ) {
		// Use default
	}
}

// Calculate next reset date
$next_reset = date( 'F j, Y', strtotime( '+1 month' ) );
?>

<div class="wrap aeowpplugin-dashboard">
    <div class="aeowpplugin-header">
        <div class="aeowpplugin-header-content">
            <h1><?php echo esc_html__( 'AEO Audit Tool', 'aeowpplugin' ); ?></h1>
            <p><?php echo esc_html__( 'Optimize your content for AI Answer Engines like Perplexity, Gemini, and SearchGPT.', 'aeowpplugin' ); ?></p>
        </div>
        <div class="aeowpplugin-header-actions" style="display: flex; align-items: center; gap: 15px;">
             <span class="aeowpplugin-version">v<?php echo AEOWPPLUGIN_VERSION; ?></span>
        </div>
    </div>

    <!-- Plan & Usage Card -->
    <div class="aeowpplugin-plan-usage-card">
        <div class="plan-info">
            <div class="plan-badge-wrap">
                <span class="plan-label"><?php echo esc_html__( 'Current Plan:', 'aeowpplugin' ); ?></span>
                <span class="plan-badge <?php echo esc_attr( $plan_badge_class ); ?>">
                    <?php echo esc_html( $plan_name ); ?>
                </span>
            </div>
            <div class="usage-info">
                <div class="usage-text">
                    <span class="usage-label"><?php echo esc_html__( 'Audits Remaining:', 'aeowpplugin' ); ?></span>
                    <span class="usage-count">
                        <strong><?php echo esc_html( $quota_remaining ); ?></strong> / <?php echo esc_html( $quota_limit ); ?>
                    </span>
                </div>
                <div class="usage-progress-bar">
                    <div class="usage-progress-fill" style="width: <?php echo esc_attr( $usage_percentage ); ?>%"></div>
                </div>
                <div class="usage-reset-info">
                    <?php echo sprintf( esc_html__( 'Resets: %s', 'aeowpplugin' ), $next_reset ); ?>
                </div>
            </div>
        </div>
        <div class="upgrade-action">
            <a href="<?php echo esc_url( $upgrade_url ); ?>" target="_blank" class="aeowpplugin-upgrade-btn">
                <span class="upgrade-icon">⚡</span>
                <?php echo esc_html__( 'Upgrade Anytime', 'aeowpplugin' ); ?>
            </a>
        </div>
    </div>

    <div class="aeowpplugin-main-grid">
        <!-- LEFT COLUMN: INPUT & SETTINGS -->
        <div class="aeowpplugin-col-left">
            <div class="aeowpplugin-card aeowpplugin-input-card">
                <h2>📊 <?php echo esc_html__( 'New Scan', 'aeowpplugin' ); ?></h2>
                <div class="aeowpplugin-scan-form">
                    <div class="aeowpplugin-field">
                        <label for="aeowpplugin-scan-url"><?php echo esc_html__( 'Target URL', 'aeowpplugin' ); ?></label>
                        <input type="url" id="aeowpplugin-scan-url" value="<?php echo esc_attr( home_url() ); ?>" placeholder="https://example.com" />
                    </div>
                    <div class="aeowpplugin-field">
                        <label for="aeowpplugin-scan-type"><?php echo esc_html__( 'Context', 'aeowpplugin' ); ?></label>
                        <select id="aeowpplugin-scan-type">
                            <option value="website">Homepage / Global</option>
                            <option value="post">Single Article / Post</option>
                        </select>
                    </div>
                    <button id="aeowpplugin-scan-btn" class="aeowpplugin-btn-primary">
                        <span>🚀</span> <?php echo esc_html__( 'Analyze Now', 'aeowpplugin' ); ?>
                    </button>
                </div>

                <div id="aeowpplugin-loading" class="aeowpplugin-loading hidden">
                    <div class="aeo-spinner"></div>
                    <p><?php echo esc_html__( 'AI is analyzing your content structure...', 'aeowpplugin' ); ?></p>
                </div>
            </div>


            <div class="aeowpplugin-card aeowpplugin-status-card <?php echo $has_results ? '' : 'hidden'; ?>" id="aeowpplugin-sidebar-status">
                <h3><?php echo esc_html__( 'Scan Summary', 'aeowpplugin' ); ?></h3>
                <div class="aeowpplugin-summary-item">
                    <label><?php echo esc_html__( 'Status', 'aeowpplugin' ); ?></label>
                    <span id="aeowpplugin-summary-status" class="aeowpplugin-status-badge <?php echo $has_results ? strtolower($scan_status) : ''; ?>">
                         <?php echo $has_results ? esc_html($scan_status) : ''; ?>
                    </span>
                </div>
                <div class="aeowpplugin-summary-item">
                    <label><?php echo esc_html__( 'Last Scan', 'aeowpplugin' ); ?></label>
                    <p id="aeowpplugin-scan-date" class="aeowpplugin-date-text">
                        <?php echo $has_results && $scan_timestamp > 0 ? date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $scan_timestamp ) : ''; ?>
                    </p>
                </div>
                <div class="aeowpplugin-summary-item">
                    <label><?php echo esc_html__( 'Target', 'aeowpplugin' ); ?></label>
                    <p id="aeowpplugin-scanned-url" class="aeowpplugin-url-text">
                        <?php echo $has_results ? esc_html($scan_url) : ''; ?>
                    </p>
                </div>
            </div>

            <!-- Professional Resources Card -->
            <div class="aeowpplugin-card aeowpplugin-resources-card">
                <h3><?php echo esc_html__( 'AEO Knowledge Hub', 'aeowpplugin' ); ?></h3>
                <div class="aeowpplugin-resource-item">
                    <div class="resource-icon">🎓</div>
                    <div class="resource-content">
                        <strong><?php echo esc_html__( 'AEO Institute', 'aeowpplugin' ); ?></strong>
                        <p><?php echo esc_html__( 'Master the fundamentals of Answer Engine Optimization.', 'aeowpplugin' ); ?></p>
                        <a href="https://aeoinstitute.com" target="_blank" class="aeowpplugin-link"><?php echo esc_html__( 'Start Learning →', 'aeowpplugin' ); ?></a>
                    </div>
                </div>
                <div class="aeowpplugin-resource-item">
                    <div class="resource-icon">🛠️</div>
                    <div class="resource-content">
                        <strong><?php echo esc_html__( 'ThinkAEO', 'aeowpplugin' ); ?></strong>
                        <p><?php echo esc_html__( 'Tools and strategic resources for AEO professionals.', 'aeowpplugin' ); ?></p>
                        <a href="https://thinkaeo.com" target="_blank" class="aeowpplugin-link"><?php echo esc_html__( 'Explore Resources →', 'aeowpplugin' ); ?></a>
                    </div>
                </div>
                <div class="aeowpplugin-resource-item">
                    <div class="resource-icon">🤝</div>
                    <div class="resource-content">
                        <strong><?php echo esc_html__( 'Hire AEO Auditor', 'aeowpplugin' ); ?></strong>
                        <p><?php echo esc_html__( 'Get professional help to fix complex AEO issues and boost visibility.', 'aeowpplugin' ); ?></p>
                        <a href="https://aeoauditor.com" target="_blank" class="aeowpplugin-link"><?php echo esc_html__( 'Hire Now →', 'aeowpplugin' ); ?></a>
                    </div>
                </div>
            </div>


        </div>

        <!-- RIGHT COLUMN: RESULTS -->
        <div class="aeowpplugin-col-right">
            <!-- Error State -->
            <div id="aeowpplugin-error" class="aeowpplugin-card aeowpplugin-error-card hidden">
                <div class="aeowpplugin-error-icon">⚠️</div>
                <h3 class="aeowpplugin-error-title"><?php echo esc_html__( 'Scan Failed', 'aeowpplugin' ); ?></h3>
                <p class="aeowpplugin-error-message"></p>
            </div>

            <!-- Empty State -->
            <div id="aeowpplugin-empty-state" class="aeowpplugin-empty-container <?php echo $has_results ? 'hidden' : ''; ?>">
                <div class="aeowpplugin-empty-icon">🔍</div>
                <h2><?php echo esc_html__( 'Ready to optimize?', 'aeowpplugin' ); ?></h2>
                <p><?php echo esc_html__( 'Enter your website URL to the left to see how AI search engines interpret your content.', 'aeowpplugin' ); ?></p>
            </div>

            <!-- Summary Results (Memory-Safe) -->
            <div id="aeowpplugin-results" class="<?php echo $has_results ? '' : 'hidden'; ?>">
                <!-- Main Score Card -->
                <div class="aeowpplugin-card aeowpplugin-score-wrap">
                    <div class="aeowpplugin-score-header">
                        <h2><?php echo esc_html__( 'Overall AEO Score', 'aeowpplugin' ); ?></h2>
                        <div class="aeowpplugin-main-score-val" id="aeowpplugin-score-num">
                            <?php echo esc_html($scan_score); ?>
                        </div>
                    </div>
                    
                    <div class="aeowpplugin-score-visual">
                        <div class="aeowpplugin-gauge-wrap">
                            <canvas id="aeowpplugin-gauge" width="200" height="200"></canvas>
                        </div>
                        <div class="aeowpplugin-breakdown">
                            <div class="breakdown-row">
                                <span><?php echo esc_html__( 'Content Logic', 'aeowpplugin' ); ?></span>
                                <div class="progress-bar"><div class="fill" id="fill-content" style="width: 0%"></div></div>
                            </div>
                            <div class="breakdown-row">
                                <span><?php echo esc_html__( 'Technical Clarity', 'aeowpplugin' ); ?></span>
                                <div class="progress-bar"><div class="fill" id="fill-technical" style="width: 0%"></div></div>
                            </div>
                            <div class="breakdown-row">
                                <span><?php echo esc_html__( 'Structural Schema', 'aeowpplugin' ); ?></span>
                                <div class="progress-bar"><div class="fill" id="fill-structure" style="width: 0%"></div></div>
                            </div>
                        </div>
                    </div>

                    <!-- Technical Details Grid (populated via AJAX) -->
                    <div id="aeowpplugin-tech-details" class="aeowpplugin-tech-grid hidden" style="margin-top: 25px; padding-top: 20px; border-top: 1px solid #eee;">
                        <!-- Populated via AJAX -->
                    </div>
                </div>

                <!-- Issues Card (populated via AJAX) -->
                <div class="aeowpplugin-card">
                    <h3><?php echo esc_html__( 'Action Required', 'aeowpplugin' ); ?></h3>
                    <div id="aeowpplugin-issues-list" class="aeowpplugin-list">
                        <?php if ($has_results && $issues_count > 0): ?>
                            <p><?php echo sprintf( esc_html__( 'Found %d issues. Click "Analyze Now" to see details.', 'aeowpplugin' ), $issues_count ); ?></p>
                        <?php else: ?>
                            <p><?php echo esc_html__( 'Run a scan to see issues.', 'aeowpplugin' ); ?></p>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Recommendations Card (populated via AJAX) -->
                <div class="aeowpplugin-card">
                    <h3><?php echo esc_html__( 'Optimization Successes', 'aeowpplugin' ); ?></h3>
                    <div id="aeowpplugin-recommendations-list" class="aeowpplugin-list">
                        <p><?php echo esc_html__( 'Run a scan to see recommendations.', 'aeowpplugin' ); ?></p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
