<?php
/**
 * AEO WP Plugin Settings Page - Minimal Version
 *
 * @package AEOWPPlugin
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Get license data with complete error handling
$license_key = get_option( 'aeowpplugin_license_key', '' );
$is_active = false;
$status = 'inactive';
$license_data = array(
	'plan' => 'free',
	'status' => 'inactive',
	'expires_at' => null,
	'quota' => array(
		'used' => 0,
		'limit' => 5,
		'period' => 'month',
	),
);

// Try to get real license data
if ( class_exists( 'AEOWPPlugin_License_Manager' ) ) {
	try {
		$license_data = AEOWPPlugin_License_Manager::get_license_data();
		$is_active = AEOWPPlugin_License_Manager::is_license_active();
		$status = $license_data['status'] ?? 'inactive';
	} catch ( Exception $e ) {
		// Use defaults
	}
}

// Helper function fallbacks
if ( ! function_exists( 'safe_get_plan_name' ) ) {
	function safe_get_plan_name( $plan ) {
		if ( function_exists( 'aeowpplugin_get_plan_name' ) ) {
			try {
				return aeowpplugin_get_plan_name( $plan );
			} catch ( Exception $e ) {
				return ucfirst( $plan );
			}
		}
		return ucfirst( $plan );
	}
}

if ( ! function_exists( 'safe_format_date' ) ) {
	function safe_format_date( $date ) {
		if ( function_exists( 'aeowpplugin_format_date' ) ) {
			try {
				return aeowpplugin_format_date( $date );
			} catch ( Exception $e ) {
				return $date ?? 'N/A';
			}
		}
		return $date ?? 'N/A';
	}
}
?>

<div class="wrap aeowpplugin-settings">
	<h1><?php echo esc_html__( 'AEO WP Settings', 'aeowpplugin' ); ?></h1>

	<div class="aeowpplugin-container">
		<!-- License Activation Section -->
		<div class="aeowpplugin-card">
			<h2><?php echo esc_html__( 'License Activation', 'aeowpplugin' ); ?></h2>
			<p class="description">
				<?php echo esc_html__( 'Activate your license key to unlock premium features and higher scan limits.', 'aeowpplugin' ); ?>
			</p>

			<div class="aeowpplugin-license-form">
				<div class="aeowpplugin-field">
					<label for="aeowpplugin-license-key"><?php echo esc_html__( 'License Key', 'aeowpplugin' ); ?></label>
					<input 
						type="text" 
						id="aeowpplugin-license-key" 
						value="<?php echo esc_attr( $license_key ); ?>" 
						placeholder="FREE-XXXX-XXXX or PRO-XXXX-XXXX"
						<?php echo $is_active ? 'readonly' : ''; ?>
					/>
					<p class="description">
						<?php echo esc_html__( 'Enter your license key. Get one from', 'aeowpplugin' ); ?> 
						<a href="https://aeowpplugin.com/pricing" target="_blank">aeowpplugin.com/pricing</a>
					</p>
					<p class="description">
						<strong><?php echo esc_html__( 'Test keys:', 'aeowpplugin' ); ?></strong> 
						FREE-TEST-001, WEEKLY-TEST-001, PRO-TEST-001, ENT-TEST-001
					</p>
				</div>

				<div class="aeowpplugin-license-actions">
					<button id="aeowpplugin-activate-license" class="button button-primary" <?php echo $is_active ? 'disabled' : ''; ?>>
						<?php echo esc_html__( 'Activate License', 'aeowpplugin' ); ?>
					</button>
					<?php if ( $is_active ): ?>
					<button id="aeowpplugin-deactivate-license" class="button">
						<?php echo esc_html__( 'Deactivate', 'aeowpplugin' ); ?>
					</button>
					<?php endif; ?>
					<span id="aeowpplugin-license-status-badge" class="aeowpplugin-badge <?php echo $is_active ? 'badge-active' : 'badge-inactive'; ?>">
						<?php echo esc_html( ucfirst( $status ) ); ?>
					</span>
				</div>

				<div id="aeowpplugin-settings-message" class="aeowpplugin-message hidden"></div>

				<?php if ( $is_active ): ?>
				<div class="aeowpplugin-license-details">
					<h3><?php echo esc_html__( 'License Details', 'aeowpplugin' ); ?></h3>
					<table class="license-details-table">
						<tr>
							<td><strong><?php echo esc_html__( 'Plan:', 'aeowpplugin' ); ?></strong></td>
							<td><?php echo esc_html( safe_get_plan_name( $license_data['plan'] ) ); ?></td>
						</tr>
						<tr>
							<td><strong><?php echo esc_html__( 'Status:', 'aeowpplugin' ); ?></strong></td>
							<td><?php echo esc_html( ucfirst( $license_data['status'] ) ); ?></td>
						</tr>
						<tr>
							<td><strong><?php echo esc_html__( 'Expires:', 'aeowpplugin' ); ?></strong></td>
							<td><?php echo esc_html( safe_format_date( $license_data['expires_at'] ) ); ?></td>
						</tr>
						<tr>
							<td><strong><?php echo esc_html__( 'Quota:', 'aeowpplugin' ); ?></strong></td>
							<td>
								<?php 
								if ( isset( $license_data['quota'] ) ) {
									echo sprintf(
										'%d / %d scans per %s',
										$license_data['quota']['used'],
										$license_data['quota']['limit'],
										$license_data['quota']['period']
									);
								} else {
									echo '0 / 5 scans per month';
								}
								?>
							</td>
						</tr>
						<tr>
							<td><strong><?php echo esc_html__( 'Version:', 'aeowpplugin' ); ?></strong></td>
							<td>
								<?php 
								$update = aeowpplugin_get_update_status();
								echo esc_html( $update['version'] );
								?>
								<span title="<?php echo esc_attr( $update['msg'] ); ?>" style="margin-left: 8px; font-size: 11px; display: inline-flex; align-items: center; gap: 4px; padding: 2px 8px; border-radius: 12px; background: <?php echo $update['status'] === 'Up to date' ? '#ecfdf5' : '#fff7ed'; ?>; color: <?php echo $update['status'] === 'Up to date' ? '#059669' : '#d97706'; ?>;">
									<span style="width: 6px; height: 6px; border-radius: 50%; background: currentColor;"></span>
									<?php echo esc_html( $update['status'] ); ?>
								</span>
							</td>
						</tr>
					</table>
				</div>
				<?php endif; ?>
			</div>
		</div>

		<!-- Account Info Section -->
		<div class="aeowpplugin-card">
			<h2><?php echo esc_html__( 'Account Information', 'aeowpplugin' ); ?></h2>

			<div class="aeowpplugin-info-section">
				<?php 
				$plan_type = safe_get_plan_name( $license_data['plan'] );
				$scans_used = $license_data['quota']['used'] ?? 0;
				$scans_limit = $license_data['quota']['limit'] ?? 5;
				$scans_remaining = max( 0, $scans_limit - $scans_used );
				$support_level = 'Community';
				if ( $license_data['plan'] === 'pro' ) $support_level = 'Priority Email';
				if ( $license_data['plan'] === 'enterprise' ) $support_level = '24/7 Priority';
				?>
				<div class="aeowpplugin-info-row">
					<div class="aeowpplugin-info-item">
						<label><?php echo esc_html__( 'Plan Type', 'aeowpplugin' ); ?></label>
						<div class="aeowpplugin-info-value"><?php echo esc_html( $plan_type ); ?></div>
					</div>
					<div class="aeowpplugin-info-item">
						<label><?php echo esc_html__( 'Scans Used', 'aeowpplugin' ); ?></label>
						<div class="aeowpplugin-info-value"><?php echo esc_html( $scans_used ); ?> / <?php echo esc_html( $scans_limit ); ?></div>
					</div>
				</div>
				<div class="aeowpplugin-info-row">
					<div class="aeowpplugin-info-item">
						<label><?php echo esc_html__( 'Scans Remaining', 'aeowpplugin' ); ?></label>
						<div class="aeowpplugin-info-value"><?php echo esc_html( $scans_remaining ); ?></div>
					</div>
					<div class="aeowpplugin-info-item">
						<label><?php echo esc_html__( 'Support Level', 'aeowpplugin' ); ?></label>
						<div class="aeowpplugin-info-value"><?php echo esc_html( $support_level ); ?></div>
					</div>
				</div>
			</div>
		</div>

		<!-- Danger Zone -->
		<div class="aeowpplugin-card aeowpplugin-danger-zone">
			<h2>⚠️ <?php echo esc_html__( 'Danger Zone', 'aeowpplugin' ); ?></h2>
			<p class="description">
				<?php echo esc_html__( 'This will clear your license key and scan results history. Note: Scan usage limits are NOT reset.', 'aeowpplugin' ); ?>
			</p>
			
			<div id="aeowpplugin-reset-message" class="aeowpplugin-message hidden" style="margin-bottom: 15px;"></div>

			<button type="button" id="aeowpplugin-reset-settings" class="aeowpplugin-btn-danger" style="padding: 10px 20px; background: #dc2626; color: #fff; border: none; border-radius: 6px; font-weight: 700; cursor: pointer;">
				<?php echo esc_html__( 'Reset License & Data', 'aeowpplugin' ); ?>
			</button>
		</div>
	</div>
</div>

<style>
.aeowpplugin-settings {
	max-width: 1200px;
}

.aeowpplugin-container {
	margin-top: 20px;
}

.aeowpplugin-card {
	background: #fff;
	border: 1px solid #ccd0d4;
	border-radius: 4px;
	padding: 20px;
	margin-bottom: 20px;
}

.aeowpplugin-card h2 {
	margin-top: 0;
	font-size: 18px;
	font-weight: 600;
}

.aeowpplugin-field {
	margin-bottom: 20px;
}

.aeowpplugin-field label {
	display: block;
	font-weight: 600;
	margin-bottom: 8px;
}

.aeowpplugin-field input[type="text"] {
	width: 100%;
	max-width: 500px;
	padding: 8px 12px;
}

.aeowpplugin-license-actions {
	display: flex;
	align-items: center;
	gap: 10px;
	margin-bottom: 20px;
}

.aeowpplugin-badge {
	padding: 4px 12px;
	border-radius: 12px;
	font-size: 12px;
	font-weight: 600;
}

.aeowpplugin-badge.badge-active {
	background: #d1fae5;
	color: #065f46;
}

.aeowpplugin-badge.badge-inactive {
	background: #f3f4f6;
	color: #6b7280;
}

.aeowpplugin-message {
	padding: 12px;
	border-radius: 4px;
	margin-bottom: 15px;
}

.aeowpplugin-message.success {
	background: #d1fae5;
	color: #065f46;
	border: 1px solid #6ee7b7;
}

.aeowpplugin-message.error {
	background: #fee2e2;
	color: #7f1d1d;
	border: 1px solid #fca5a5;
}

.aeowpplugin-message.hidden {
	display: none;
}

.aeowpplugin-license-details {
	margin-top: 20px;
	padding-top: 20px;
	border-top: 1px solid #e5e7eb;
}

.license-details-table {
	width: 100%;
	max-width: 600px;
}

.license-details-table td {
	padding: 8px 0;
}

.license-details-table td:first-child {
	width: 150px;
}

.aeowpplugin-info-section {
	margin-top: 15px;
}

.aeowpplugin-info-row {
	display: grid;
	grid-template-columns: 1fr 1fr;
	gap: 20px;
	margin-bottom: 15px;
}

.aeowpplugin-info-item label {
	display: block;
	font-size: 12px;
	color: #6b7280;
	margin-bottom: 4px;
}

.aeowpplugin-info-value {
	font-size: 16px;
	font-weight: 600;
	color: #111827;
}

.aeowpplugin-danger-zone h2 {
	color: #dc2626;
}
</style>

<script>
/**
 * Direct Script for Reset Button - Guarantees execution 
 */
jQuery(document).ready(function($) {
    console.log('AEO Settings: Reset script initialized');
    
    $(document).on('click', '#aeowpplugin-reset-settings', function(e) {
        e.preventDefault();
        
        console.log('AEO Settings: Reset clicked');
        
        if (!confirm('Are you sure you want to reset your License and Scan Data? This will NOT reset your scan limits.')) {
            return;
        }

        const $btn = $(this);
        const $msg = $('#aeowpplugin-reset-message');
        
        $btn.prop('disabled', true).text('Working...');
        $msg.addClass('hidden');

        $.ajax({
            url: aeowppluginData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'aeo_reset_settings',
                nonce: aeowppluginData.nonce
            },
            success: function(response) {
                console.log('AEO Settings: Reset response', response);
                if (response.success) {
                    $msg.text(response.data.message || 'Success! Settings reset.').attr('class', 'aeowpplugin-message success').removeClass('hidden');
                    setTimeout(function() {
                        window.location.reload();
                    }, 1500);
                } else {
                    $msg.text(response.data.message || 'Error').attr('class', 'aeowpplugin-message error').removeClass('hidden');
                    $btn.prop('disabled', false).text('Reset License & Data');
                }
            },
            error: function() {
                $msg.text('Connection error.').attr('class', 'aeowpplugin-message error').removeClass('hidden');
                $btn.prop('disabled', false).text('Reset License & Data');
            }
        });
    });
});
</script>
