<?php
/**
 * Plugin Name: AEO WP Plugin
 * Plugin URI: https://aeowpplugin.com
 * Description: Check if your WordPress site or content is Answer Engine Optimization (AEO) friendly with actionable recommendations.
 * Version: 2.0.0
 * Author: AEO WP Plugin
 * Author URI: https://aeowpplugin.com
 * License: GPL v2 or later
 * Text Domain: aeowpplugin
 * Requires at least: 5.9
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Define plugin constants.
define( 'AEOWPPLUGIN_VERSION', '2.0.0' );
define( 'AEOWPPLUGIN_PATH', plugin_dir_path( __FILE__ ) );
define( 'AEOWPPLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'AEOWPPLUGIN_BASENAME', plugin_basename( __FILE__ ) );
define( 'AEOWPPLUGIN_API_BASE_URL', 'https://aeowpplugin.com/api' );

/**
 * Main plugin class
 */
class AEOWPPlugin {

	private static $instance;


	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	public function __construct() {
		$this->includes();
		$this->init_hooks();
	}

	private function includes() {
		// Core classes - load in correct order
		$files = array(
			'includes/class-license-manager.php',
			'includes/class-quota-manager.php',
			'includes/class-api-client.php',
			'includes/class-analyzer.php',
			'includes/class-updater.php',
			'includes/helpers.php',
			'includes/api-handler.php',
			'includes/rate-limit.php',
		);

		foreach ( $files as $file ) {
			$filepath = AEOWPPLUGIN_PATH . $file;
			if ( file_exists( $filepath ) ) {
				require_once $filepath;
			} else {
				// Log error but don't break the plugin
				error_log( 'AEO WP Plugin: Missing file - ' . $file );
			}
		}
	}

	private function init_hooks() {
		add_action( 'admin_menu', array( $this, 'register_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );

		// AJAX Handlers
		add_action( 'wp_ajax_aeo_run_scan', array( 'AEOWPPlugin_API_Handler', 'handle_ajax_scan' ) );
		add_action( 'wp_ajax_aeo_verify_license', array( 'AEOWPPlugin_API_Handler', 'handle_ajax_verify_license' ) );
		add_action( 'wp_ajax_aeo_activate_license', array( $this, 'handle_ajax_activate_license' ) );
		add_action( 'wp_ajax_aeo_deactivate_license', array( $this, 'handle_ajax_deactivate_license' ) );
		add_action( 'wp_ajax_aeo_reset_settings', array( $this, 'handle_ajax_reset_settings' ) );

		// Initialization
		new AEOWPPlugin_Updater( AEOWPPLUGIN_VERSION, 'aeowpplugin' );
		
		add_filter( 'plugin_action_links_' . AEOWPPLUGIN_BASENAME, array( $this, 'add_plugin_action_links' ) );
		
		// Activation / Deactivation
		register_activation_hook( __FILE__, array( $this, 'activate_plugin' ) );
	}

	public function register_admin_menu() {
		add_menu_page(
			__( 'AEO WP', 'aeowpplugin' ),
			__( 'AEO WP', 'aeowpplugin' ),
			'manage_options',
			'aeowpplugin',
			array( $this, 'render_dashboard' ),
			'dashicons-chart-line',
			25
		);

		add_submenu_page(
			'aeowpplugin',
			__( 'Dashboard', 'aeowpplugin' ),
			__( 'Dashboard', 'aeowpplugin' ),
			'manage_options',
			'aeowpplugin',
			array( $this, 'render_dashboard' )
		);

		add_submenu_page(
			'aeowpplugin',
			__( 'Settings', 'aeowpplugin' ),
			__( 'Settings', 'aeowpplugin' ),
			'manage_options',
			'aeowpplugin-settings',
			array( $this, 'render_settings' )
		);
	}

	/**
	 * Add Settings link to plugin action links
	 */
	public function add_plugin_action_links( $links ) {
		$settings_link = array(
			'<a href="' . admin_url( 'admin.php?page=aeowpplugin-settings' ) . '">' . __( 'Settings', 'aeowpplugin' ) . '</a>',
		);
		return array_merge( $settings_link, $links );
	}

	public function render_dashboard() {
		try {
			$dashboard_file = AEOWPPLUGIN_PATH . 'admin/dashboard.php';
			if ( file_exists( $dashboard_file ) ) {
				include $dashboard_file;
			} else {
				echo '<div class="wrap"><h1>Error</h1><p>Dashboard file not found: ' . esc_html( $dashboard_file ) . '</p></div>';
			}
		} catch ( Exception $e ) {
			echo '<div class="wrap"><h1>Error</h1><p>Dashboard error: ' . esc_html( $e->getMessage() ) . '</p></div>';
		}
	}

	public function render_settings() {
		try {
			$settings_file = AEOWPPLUGIN_PATH . 'admin/settings.php';
			if ( file_exists( $settings_file ) ) {
				include $settings_file;
			} else {
				echo '<div class="wrap"><h1>Error</h1><p>Settings file not found: ' . esc_html( $settings_file ) . '</p></div>';
			}
		} catch ( Exception $e ) {
			echo '<div class="wrap"><h1>Error</h1><p>Settings error: ' . esc_html( $e->getMessage() ) . '</p></div>';
		}
	}

	public function enqueue_admin_scripts( $hook ) {
		// Only load on our plugin pages
		if ( strpos( $hook, 'aeowpplugin' ) === false ) {
			return;
		}

		// CSS
		wp_enqueue_style(
			'aeowpplugin-admin',
			AEOWPPLUGIN_URL . 'assets/css/admin.css',
			array(),
			AEOWPPLUGIN_VERSION
		);

		// Chart.js (External)
		wp_enqueue_script(
			'aeowpplugin-chart-js',
			'https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.js',
			array(),
			'4.4.0',
			true
		);

		// Main Admin JS
		wp_enqueue_script(
			'aeowpplugin-admin',
			AEOWPPLUGIN_URL . 'assets/js/admin.js',
			array( 'jquery' ),
			AEOWPPLUGIN_VERSION,
			true
		);

		// Localize data for JS
		wp_localize_script(
			'aeowpplugin-admin',
			'aeowppluginData',
			array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'aeo_scan_nonce' ),
				'homeUrl' => home_url(),
			)
		);
	}

	public function activate_plugin() {
		if ( ! get_option( 'aeowpplugin_api_key' ) ) {
			add_option( 'aeowpplugin_api_key', '' );
		}

		// CRITICAL: Clean up old large scan data to prevent memory exhaustion
		$this->cleanup_old_scan_data();
	}

	/**
	 * Clean up old scan data that causes memory issues
	 */
	private function cleanup_old_scan_data() {
		// Delete old full scan results (can be huge and cause memory exhaustion)
		delete_option( 'aeowpplugin_last_scan' );
		
		// Remove any cached scan data
		wp_cache_delete( 'aeowpplugin_last_scan' );
		wp_cache_delete( 'aeowpplugin_scan_cache' );
		
		// Log cleanup
		error_log( 'AEO WP Plugin: Cleaned up old scan data to prevent memory issues' );
	}

	/**
	 * Handle AJAX license activation
	 */
	public function handle_ajax_activate_license() {
		check_ajax_referer( 'aeo_scan_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ) );
		}

		$license_key = isset( $_POST['licenseKey'] ) ? sanitize_text_field( $_POST['licenseKey'] ) : '';

		if ( empty( $license_key ) ) {
			wp_send_json_error( array( 'message' => 'License key is required' ) );
		}

		$result = AEOWPPlugin_License_Manager::activate_license( $license_key );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		wp_send_json_success( $result );
	}

	/**
	 * Handle AJAX license deactivation
	 */
	public function handle_ajax_deactivate_license() {
		check_ajax_referer( 'aeo_scan_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ) );
		}

		AEOWPPlugin_License_Manager::deactivate_license();

		wp_send_json_success( array(
			'message' => 'License deactivated successfully',
		) );
	}

	/**
	 * Handle AJAX settings reset
	 */
	public function handle_ajax_reset_settings() {
		check_ajax_referer( 'aeo_scan_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ) );
		}

		// List of options to delete
		$options = array(
			'aeowpplugin_license_key',
			'aeowpplugin_license_data',
			'aeowpplugin_last_scan_summary',
			'aeowpplugin_scan_history',
			'aeowpplugin_api_key',
			'aeowpplugin_last_scan',
		);

		foreach ( $options as $option ) {
			delete_option( $option );
			wp_cache_delete( $option );
		}

		wp_send_json_success( array(
			'message' => 'All settings have been reset successfully.',
		) );
	}
}

// Initialize
add_action( 'plugins_loaded', array( 'AEOWPPlugin', 'get_instance' ) );
