/**
 * AEO WP Plugin - Premium Admin JS
 */
(function ($) {
    'use strict';

    $(document).ready(function () {
        console.log('AEO WP Plugin: Admin JS Loaded');

        const $loadingEl = $('#aeowpplugin-loading');
        const $errorEl = $('#aeowpplugin-error');
        const $resultsEl = $('#aeowpplugin-results');
        const $emptyStateEl = $('#aeowpplugin-empty-state');
        const $sidebarStatus = $('#aeowpplugin-sidebar-status');
        const $settingsMsg = $('#aeowpplugin-settings-message');

        let gaugeChart = null;

        // --- 1. GLOBAL DELEGATED LISTENERS --- (More robust for WP)

        // Reset Settings Handler
        $(document).on('click', '#aeowpplugin-reset-settings', function (e) {
            e.preventDefault();
            console.log('AEO WP Plugin: Reset button clicked');

            if (!confirm('Are you sure you want to reset your License and Scan Data? This will NOT reset your scan limits.')) {
                return;
            }

            const $btn = $(this);
            const $specificResetMsg = $('#aeowpplugin-reset-message');
            const originalText = $btn.text();

            $btn.prop('disabled', true).text('Resetting...');

            $.post(aeowppluginData.ajaxUrl, {
                action: 'aeo_reset_settings',
                nonce: aeowppluginData.nonce
            }, function (response) {
                console.log('AEO WP Plugin: Reset response', response);
                if (response.success) {
                    const msg = response.data.message || 'Settings reset successfully.';
                    if ($specificResetMsg.length) {
                        $specificResetMsg.text(msg).attr('class', 'aeowpplugin-message success').removeClass('hidden');
                    } else {
                        showSettingsMessage(msg, 'success');
                    }
                    setTimeout(() => location.reload(), 1500);
                } else {
                    const errMsg = response.data.message || 'Reset failed.';
                    if ($specificResetMsg.length) {
                        $specificResetMsg.text(errMsg).attr('class', 'aeowpplugin-message error').removeClass('hidden');
                    } else {
                        showSettingsMessage(errMsg, 'error');
                    }
                    $btn.prop('disabled', false).text('Reset License & Data');
                }
            }).fail(function () {
                const connErr = 'Connection error. Please try again.';
                if ($specificResetMsg.length) {
                    $specificResetMsg.text(connErr).attr('class', 'aeowpplugin-message error').removeClass('hidden');
                } else {
                    showSettingsMessage(connErr, 'error');
                }
                $btn.prop('disabled', false).text('Reset License & Data');
            });
        });

        // Activate License Handler
        $(document).on('click', '#aeowpplugin-activate-license', function (e) {
            e.preventDefault();
            const $btn = $(this);
            const $keyInput = $('#aeowpplugin-license-key');
            const key = $keyInput.val().trim();

            if (!key) {
                showSettingsMessage('Please enter a license key.', 'error');
                return;
            }

            const originalText = $btn.text();
            $btn.prop('disabled', true).text('Activating...');

            $.post(aeowppluginData.ajaxUrl, {
                action: 'aeo_activate_license',
                nonce: aeowppluginData.nonce,
                licenseKey: key
            }, function (response) {
                if (response.success) {
                    showSettingsMessage('License activated! Reloading...', 'success');
                    setTimeout(() => location.reload(), 1000);
                } else {
                    showSettingsMessage(response.data.message || 'Activation failed.', 'error');
                    $btn.prop('disabled', false).text(originalText);
                }
            }).fail(function () {
                showSettingsMessage('Connection error.', 'error');
                $btn.prop('disabled', false).text(originalText);
            });
        });

        // Deactivate License Handler
        $(document).on('click', '#aeowpplugin-deactivate-license', function (e) {
            e.preventDefault();
            if (!confirm('Deactivate your license?')) return;

            const $btn = $(this);
            const originalText = $btn.text();
            $btn.prop('disabled', true).text('Deactivating...');

            $.post(aeowppluginData.ajaxUrl, {
                action: 'aeo_deactivate_license',
                nonce: aeowppluginData.nonce
            }, function (response) {
                if (response.success) {
                    showSettingsMessage('License deactivated. Reloading...', 'success');
                    setTimeout(() => location.reload(), 1000);
                } else {
                    showSettingsMessage('Deactivation failed.', 'error');
                    $btn.prop('disabled', false).text(originalText);
                }
            }).fail(function () {
                showSettingsMessage('Connection error.', 'error');
                $btn.prop('disabled', false).text(originalText);
            });
        });

        // Scan/Analyze Handler
        $(document).on('click', '#aeowpplugin-scan-btn', function (e) {
            e.preventDefault();
            performScan();
        });

        // --- 2. INITIALIZATION ---
        if ($resultsEl.length && !$resultsEl.hasClass('hidden')) {
            const score = parseInt($('#aeowpplugin-score-num').text(), 10) || 0;
            const status = $('#aeowpplugin-summary-status').text().trim();
            if (score > 0) updateGauge(score, status);
        }

        // --- 3. HELPER FUNCTIONS ---

        function performScan() {
            const $scanBtn = $('#aeowpplugin-scan-btn');
            const url = $('#aeowpplugin-scan-url').val().trim();
            const type = $('#aeowpplugin-scan-type').val();

            if (!url) {
                showError('Please enter a valid URL.');
                return;
            }

            $scanBtn.prop('disabled', true).addClass('scanning');
            $resultsEl.addClass('hidden');
            $emptyStateEl.addClass('hidden');
            $errorEl.addClass('hidden');
            $sidebarStatus.addClass('hidden');
            $loadingEl.removeClass('hidden');

            $.post(aeowppluginData.ajaxUrl, {
                action: 'aeo_run_scan',
                nonce: aeowppluginData.nonce,
                url: url,
                type: type
            }, function (response) {
                if (response.success) {
                    renderFullResults(response.data);
                    if (response.data.quota) updateQuotaDisplay(response.data.quota);
                } else {
                    if (response.data.quota && response.data.upgrade_url) {
                        showQuotaError(response.data.message, response.data.upgrade_url);
                    } else {
                        showError(response.data.message || 'Scan failed.');
                    }
                }
            }).fail(function () {
                showError('Server error. Please check your connection.');
            }).always(function () {
                $scanBtn.prop('disabled', false).removeClass('scanning');
                $loadingEl.addClass('hidden');
            });
        }

        function renderFullResults(data) {
            $('#aeowpplugin-score-num').text(data.score);
            updateGauge(data.score, data.status);

            if (data.breakdown) {
                $('#fill-content').css('width', data.breakdown.content + '%');
                $('#fill-technical').css('width', data.breakdown.technical + '%');
                $('#fill-structure').css('width', data.breakdown.structure + '%');
            }

            const $techGrid = $('#aeowpplugin-tech-details');
            $techGrid.empty();
            if (data.technical_details) {
                Object.entries(data.technical_details).forEach(([key, val]) => {
                    $techGrid.append(`<div class="tech-item"><span class="tech-label">${escapeHtml(key)}</span><span class="tech-val">${escapeHtml(val)}</span></div>`);
                });
                $techGrid.removeClass('hidden');
            }

            $('#aeowpplugin-scanned-url').text(data.url);
            $('#aeowpplugin-scan-date').text(new Date().toLocaleString());
            $('#aeowpplugin-summary-status').text(data.status).attr('class', 'aeowpplugin-status-badge ' + data.status.toLowerCase());

            const $issuesList = $('#aeowpplugin-issues-list');
            $issuesList.empty();
            if (data.issues && data.issues.length > 0) {
                data.issues.forEach(issue => {
                    let solutionHtml = '';
                    if (issue.solution) {
                        solutionHtml = `
                            <div class="issue-solution">
                                <strong>💡 How to Fix:</strong>
                                <p>${escapeHtml(issue.solution)}</p>
                            </div>
                        `;
                    }
                    $issuesList.append(`<div class="aeowpplugin-item-box issue ${issue.severity}"><div class="item-icon">⚠️</div><div class="item-main"><div class="item-header"><strong>${escapeHtml(issue.title)}</strong><span class="severity-badge ${issue.severity}">${issue.severity.toUpperCase()}</span></div><div class="item-content"><p class="issue-instruction">${escapeHtml(issue.description)}</p>${solutionHtml}</div></div></div>`);
                });
            } else {
                $issuesList.append('<p class="aeowpplugin-empty-msg">No issues found. Excellent work!</p>');
            }

            const $recList = $('#aeowpplugin-recommendations-list');
            $recList.empty();
            if (data.recommendations && data.recommendations.length > 0) {
                data.recommendations.forEach(rec => {
                    $recList.append(`<div class="aeowpplugin-item-box recommendation"><div class="item-icon">✅</div><div class="item-main"><div class="item-header"><strong>${escapeHtml(rec.title)}</strong></div><div class="item-content"><p>${escapeHtml(rec.fix)}</p></div></div></div>`);
                });
            }

            $resultsEl.removeClass('hidden');
            $sidebarStatus.removeClass('hidden');
            $emptyStateEl.addClass('hidden');
        }

        function updateGauge(score, status) {
            const ctx = document.getElementById('aeowpplugin-gauge');
            if (!ctx || typeof Chart === 'undefined') return;

            if (gaugeChart) gaugeChart.destroy();

            const colors = { 'Excellent': '#10b981', 'Good': '#6366f1', 'Average': '#f59e0b', 'Poor': '#ef4444' };
            const color = colors[status] || colors['Average'];

            gaugeChart = new Chart(ctx, {
                type: 'doughnut',
                data: {
                    datasets: [{
                        data: [score, 100 - score],
                        backgroundColor: [color, '#e2e8f0'],
                        borderWidth: 0,
                        circumference: 180,
                        rotation: 270,
                    }]
                },
                options: { cutout: '85%', responsive: true, maintainAspectRatio: false, plugins: { legend: { display: false }, tooltip: { enabled: false } } }
            });
        }

        function showError(message) {
            $errorEl.find('.aeowpplugin-error-message').text(message);
            $errorEl.removeClass('hidden');
            $resultsEl.addClass('hidden');
            $sidebarStatus.addClass('hidden');
        }

        function showQuotaError(message, upgradeUrl) {
            $errorEl.find('.aeowpplugin-error-message').html(escapeHtml(message) + '<br><br><a href="' + upgradeUrl + '" target="_blank" class="aeowpplugin-upgrade-link-btn" style="display: inline-block; margin-top: 10px; padding: 10px 20px; background: #6366f1; color: white; text-decoration: none; border-radius: 6px; font-weight: 500;">⚡ Upgrade Now</a>');
            $errorEl.removeClass('hidden');
            $resultsEl.addClass('hidden');
        }

        function updateQuotaDisplay(quota) {
            $('.usage-count strong').text(quota.remaining);
            if (quota.limit > 0) {
                $('.usage-progress-fill').css('width', Math.min(100, (quota.used / quota.limit) * 100) + '%');
            }
        }

        function showSettingsMessage(msg, type) {
            if (!$settingsMsg.length) return;
            $settingsMsg.text(msg).attr('class', 'aeowpplugin-message ' + type).removeClass('hidden');
            setTimeout(() => $settingsMsg.addClass('hidden'), 5000);
        }

        function escapeHtml(text) {
            if (typeof text !== 'string') return text;
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    });

})(jQuery);
