'use client';

/**
 * Gutenberg Sidebar for AEO WP Plugin
 */

// Import wp and aeowppluginGutenberg variables
const wp = window.wp;
const aeowppluginGutenberg = window.aeowppluginGutenberg;

const { registerPlugin } = wp.plugins;
const { PluginSidebar } = wp.editPost;
const { useEffect, useState } = wp.element;
const { Button, Panel, PanelBody, Spinner, ExternalLink } = wp.components;
const apiFetch = wp.apiFetch;

// Main Sidebar Component
const AEOWPGutenbergSidebar = () => {
	const [score, setScore] = useState(null);
	const [status, setStatus] = useState(null);
	const [issues, setIssues] = useState([]);
	const [recommendations, setRecommendations] = useState([]);
	const [isLoading, setIsLoading] = useState(false);
	const [error, setError] = useState(null);
	const [hasApiKey, setHasApiKey] = useState(false);

	// Check if API key is configured
	useEffect(() => {
		checkApiKey();
	}, []);

	const checkApiKey = async () => {
		try {
			const response = await apiFetch({
				path: '/aeowpplugin/v1/settings',
				method: 'GET',
				headers: {
					'X-WP-Nonce': aeowppluginGutenberg.nonce,
				},
			});
			setHasApiKey(response.has_key);
		} catch (err) {
			console.error('Error checking API key:', err);
		}
	};

	const performScan = async () => {
		if (!hasApiKey) {
			setError('API key not configured. Please go to Settings.');
			return;
		}

		setIsLoading(true);
		setError(null);

		try {
			const response = await apiFetch({
				path: '/aeowpplugin/v1/gutenberg-scan',
				method: 'POST',
				headers: {
					'X-WP-Nonce': aeowppluginGutenberg.nonce,
				},
				data: {
					url: aeowppluginGutenberg.postPermalink,
				},
			});

			setScore(response.score);
			setStatus(response.status);
			setIssues(response.issues || []);
			setRecommendations(response.recommendations || []);
			setError(null);
		} catch (err) {
			setError(err.message || 'Failed to scan post');
			setScore(null);
			setStatus(null);
		} finally {
			setIsLoading(false);
		}
	};

	const getStatusColor = (status) => {
		const colors = {
			'Excellent': '#10b981',
			'Good': '#3b82f6',
			'Average': '#f59e0b',
			'Poor': '#ef4444',
		};
		return colors[status] || '#6b7280';
	};

	const getSeverityColor = (severity) => {
		const colors = {
			'high': '#ef4444',
			'medium': '#f59e0b',
			'low': '#3b82f6',
		};
		return colors[severity] || '#6b7280';
	};

	return (
		<PluginSidebar name="aeowpplugin-sidebar" title="AEO WP">
			<PanelBody>
				{!hasApiKey && (
					<div style={{ 
						padding: '16px', 
						backgroundColor: '#fef3c7', 
						border: '1px solid #fcd34d',
						borderRadius: '4px',
						marginBottom: '16px',
						color: '#92400e',
						fontSize: '14px',
					}}>
						<p style={{ margin: '0 0 8px 0' }}>
							<strong>API Key Required</strong>
						</p>
						<p style={{ margin: 0 }}>
							Configure your API key in Settings to use AEO scanning.
						</p>
					</div>
				)}

				<div style={{ marginBottom: '16px' }}>
					<Button 
						isPrimary 
						onClick={performScan}
						disabled={isLoading || !hasApiKey}
						style={{ width: '100%' }}
					>
						{isLoading ? (
							<>
								<Spinner style={{ marginRight: '8px' }} />
								Scanning...
							</>
						) : (
							'Scan This Post'
						)}
					</Button>
				</div>

				{error && (
					<div style={{
						padding: '12px',
						backgroundColor: '#fee2e2',
						border: '1px solid #fca5a5',
						borderRadius: '4px',
						color: '#7f1d1d',
						fontSize: '14px',
						marginBottom: '16px',
					}}>
						{error}
					</div>
				)}

				{score !== null && (
					<>
						{/* Score Display */}
						<div style={{
							textAlign: 'center',
							padding: '20px',
							backgroundColor: '#f3f4f6',
							borderRadius: '4px',
							marginBottom: '16px',
						}}>
							<div style={{
								fontSize: '40px',
								fontWeight: 'bold',
								color: getStatusColor(status),
								marginBottom: '8px',
							}}>
								{score}
							</div>
							<div style={{
								fontSize: '14px',
								color: '#6b7280',
								marginBottom: '12px',
							}}>
								{status}
							</div>
							<div style={{
								fontSize: '12px',
								color: '#9ca3af',
							}}>
								Last scan: {new Date().toLocaleDateString()}
							</div>
						</div>

						{/* Issues */}
						{issues.length > 0 && (
							<div style={{ marginBottom: '16px' }}>
								<h4 style={{ marginTop: 0, marginBottom: '12px', color: '#111827' }}>
									Issues Found ({issues.length})
								</h4>
								<div style={{ display: 'flex', flexDirection: 'column', gap: '8px' }}>
									{issues.map((issue, idx) => (
										<div 
											key={idx}
											style={{
												padding: '12px',
												borderLeft: `4px solid ${getSeverityColor(issue.severity)}`,
												backgroundColor: '#f9fafb',
												borderRadius: '4px',
												fontSize: '13px',
											}}
										>
											<div style={{
												fontWeight: 600,
												marginBottom: '4px',
												color: '#111827',
											}}>
												{issue.title}
											</div>
											<div style={{
												color: '#6b7280',
												fontSize: '12px',
											}}>
												{issue.description}
											</div>
										</div>
									))}
								</div>
							</div>
						)}

						{/* Recommendations */}
						{recommendations.length > 0 && (
							<div>
								<h4 style={{ marginTop: 0, marginBottom: '12px', color: '#111827' }}>
									Recommendations
								</h4>
								<div style={{ display: 'flex', flexDirection: 'column', gap: '8px' }}>
									{recommendations.map((rec, idx) => (
										<div 
											key={idx}
											style={{
												padding: '12px',
												borderLeft: '4px solid #3b82f6',
												backgroundColor: '#f9fafb',
												borderRadius: '4px',
												fontSize: '13px',
											}}
										>
											<div style={{
												fontWeight: 600,
												marginBottom: '4px',
												color: '#111827',
											}}>
												✓ {rec.title}
											</div>
											<div style={{
												color: '#6b7280',
												fontSize: '12px',
											}}>
												{rec.fix}
											</div>
										</div>
									))}
								</div>
							</div>
						)}
					</>
				)}
			</PanelBody>
		</PluginSidebar>
	);
};

// Register the plugin
registerPlugin('aeowpplugin-sidebar', {
	render: AEOWPGutenbergSidebar,
});
