<?php
/**
 * API Handler for AEO WP Plugin
 *
 * @package AEOWPPlugin
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AEOWPPlugin_API_Handler {

	/**
	 * Handle AJAX Scan Request
	 */
	public static function handle_ajax_scan() {
		check_ajax_referer( 'aeo_scan_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ) );
		}

		// 1. Check license status
		$license_data = AEOWPPlugin_License_Manager::get_license_data();
		
		// 2. Check quota
		$quota_check = AEOWPPlugin_Quota_Manager::can_scan();
		
		if ( ! $quota_check['allowed'] ) {
			wp_send_json_error( array(
				'message'     => $quota_check['message'],
				'quota'       => $quota_check['quota'],
				'upgrade_url' => aeowpplugin_get_upgrade_url(),
			) );
		}

		$url = isset( $_POST['url'] ) ? esc_url_raw( $_POST['url'] ) : '';
		
		if ( empty( $url ) ) {
			wp_send_json_error( array( 'message' => 'URL is required' ) );
		}

		// Perform a "Live" Analysis
		$analysis = self::analyze_url( $url );

		// Propagate Errors
		if ( is_wp_error( $analysis ) ) {
			wp_send_json_error( array( 'message' => $analysis->get_error_message() ) );
		}

		// Increment usage counter
		AEOWPPlugin_Quota_Manager::increment_usage();

		// Return the results with updated quota
		$updated_quota = AEOWPPlugin_Quota_Manager::get_quota_status();
		$analysis['quota'] = $updated_quota;

		wp_send_json_success( $analysis );
	}

	/**
	 * Handle AJAX License Verification
	 */
	public static function handle_ajax_verify_license() {
		check_ajax_referer( 'aeo_scan_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ) );
		}

		$license_key = isset( $_POST['licenseKey'] ) ? sanitize_text_field( $_POST['licenseKey'] ) : '';

		if ( empty( $license_key ) ) {
			wp_send_json_error( array( 'message' => 'License key is required' ) );
		}

		// Use License Manager to activate
		$result = AEOWPPlugin_License_Manager::activate_license( $license_key );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		// Return license data with quota info
		wp_send_json_success( $result );
	}

	/**
	 * Analyze the URL (Remote API Call with Fallback)
	 */
	private static function analyze_url( $url ) {
		$license_key = get_option( 'aeowpplugin_license_key' );

		// Try API first
		$api_response = AEOWPPlugin_API_Client::perform_scan( $url, $license_key );

		// If API works, use it
		if ( ! is_wp_error( $api_response ) && isset( $api_response['success'] ) && $api_response['success'] ) {
			$result = $api_response['data'];
		} else {
			// Fallback to mock response
			$mock_response = AEOWPPlugin_API_Client::mock_scan_response( $url );
			$result = $mock_response['data'];
		}

		// CRITICAL: Check response size to prevent memory exhaustion
		$result_json = wp_json_encode( $result );
		$result_size = strlen( $result_json );
		
		// If response is > 1MB, log error and abort
		if ( $result_size > 1048576 ) { // 1MB = 1048576 bytes
			error_log( 'AEO WP Plugin: Scan response too large (' . round( $result_size / 1048576, 2 ) . 'MB). Aborting to prevent memory exhaustion.' );
			return new WP_Error( 'response_too_large', 'Scan response exceeded 1MB limit. Please contact support.' );
		}

		// Store ONLY summary data (not full results) to prevent memory issues
		$summary = array(
			'url'         => $url,
			'timestamp'   => current_time( 'timestamp' ),
			'score'       => isset( $result['score'] ) ? intval( $result['score'] ) : 0,
			'status'      => isset( $result['status'] ) ? sanitize_text_field( $result['status'] ) : 'Unknown',
			'issues_count' => isset( $result['issues'] ) ? count( $result['issues'] ) : 0,
		);

		// Store summary only (not full result)
		update_option( 'aeowpplugin_last_scan_summary', $summary, false ); // false = no autoload

		// Keep scan history (max 10 entries) - store only summaries
		self::add_to_scan_history( $summary );

		return $result;
	}

	/**
	 * Add scan to history (max 10 entries)
	 */
	private static function add_to_scan_history( $summary ) {
		$history = get_option( 'aeowpplugin_scan_history', array() );
		
		// Add new scan to beginning
		array_unshift( $history, $summary );
		
		// Keep only last 10 scans
		$history = array_slice( $history, 0, 10 );
		
		// Store without autoload to prevent memory issues
		update_option( 'aeowpplugin_scan_history', $history, false );
	}
}
