<?php
/**
 * AEOWPPlugin Analyzer
 * Performs real-time SEO and technical analysis of a URL
 *
 * @package AEOWPPlugin
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AEOWPPlugin_Analyzer {

	/**
	 * Run analysis on a URL
	 *
	 * @param string $url The URL to analyze
	 * @return array Analysis results
	 */
	public static function run_analysis( $url ) {
		$start_time = microtime( true );

		// 1. Fetch the content
		$response = wp_remote_get( $url, array(
			'timeout'    => 15,
			'user-agent' => 'AEO-WP-Plugin/2.0.0; ' . home_url(),
		) );

		if ( is_wp_error( $response ) ) {
			return self::get_error_response( $url, $response->get_error_message() );
		}

		$html = wp_remote_retrieve_body( $response );
		$code = wp_remote_retrieve_response_code( $response );
		$ttfb = round( ( microtime( true ) - $start_time ) * 1000 ); // ms

		if ( $code !== 200 ) {
			return self::get_error_response( $url, "Site returned HTTP $code" );
		}

		// 2. Perform Analysis
		$checks = array(
			'technical' => self::check_technical( $url, $response, $html, $ttfb ),
			'content'   => self::check_content( $html ),
			'structure' => self::check_structure( $html ),
		);

		// 3. Calculate Scores
		$tech_score      = $checks['technical']['score'];
		$content_score   = $checks['content']['score'];
		$structure_score = $checks['structure']['score'];

		// Weighted Score: Content 40%, Technical 30%, Structure 30%
		$total_score = round( ( $content_score * 0.4 ) + ( $tech_score * 0.3 ) + ( $structure_score * 0.3 ) );

		// 4. Determine Status
		$status = 'Average';
		if ( $total_score >= 85 ) $status = 'Excellent';
		elseif ( $total_score >= 70 ) $status = 'Good';
		elseif ( $total_score < 50 ) $status = 'Poor';

		// 5. Build Result
		return array(
			'url'               => $url,
			'score'             => $total_score,
			'status'            => $status,
			'breakdown'         => array(
				'content'   => $content_score,
				'technical' => $tech_score,
				'structure' => $structure_score,
			),
			'technical_details' => array_merge(
				$checks['technical']['details'],
				$checks['content']['details'],
				$checks['structure']['details']
			),
			'issues'            => array_merge(
				$checks['technical']['issues'],
				$checks['content']['issues'],
				$checks['structure']['issues']
			),
			'recommendations'   => array_merge(
				$checks['technical']['recs'],
				$checks['content']['recs'],
				$checks['structure']['recs']
			),
		);
	}

	private static function check_technical( $url, $response, $html, $ttfb ) {
		$score   = 100;
		$details = array();
		$issues  = array();
		$recs    = array();

		// HTTPS check
		$is_https = ( strpos( $url, 'https://' ) === 0 );
		$details['SSL Certificate'] = $is_https ? 'Valid' : 'Missing/Insecure';
		if ( ! $is_https ) {
			$score -= 20;
			$issues[] = array(
				'title'       => 'Your Site is Not Secure (No HTTPS)',
				'description' => 'Your website is using an insecure connection. AI search engines and users prefer secure sites.',
				'solution'    => 'Install an SSL certificate through your hosting provider and redirect traffic to the https:// version of your site.',
				'severity'    => 'high'
			);
		} else {
			$recs[] = array('title' => 'SSL Enabled', 'fix' => 'Your site uses HTTPS, which is positive for AEO.');
		}

		// Response time (TTFB)
		$details['Page Load (TTFB)'] = $ttfb . 'ms';
		if ( $ttfb > 1000 ) {
			$score -= 15;
			$issues[] = array(
				'title'       => 'Server is responding slowly',
				'description' => "Your server took $ttfb ms to respond. Speed is a critical factor for AI search visibility.",
				'solution'    => 'Use a caching plugin, optimize your database, or consider upgrading to faster hosting.',
				'severity'    => 'medium'
			);
		}

		// Viewport Meta (Mobile Friendly)
		$has_viewport = ( preg_match( '/<meta[^>]+name=["\']viewport["\']/i', $html ) );
		$details['Mobile Friendly'] = $has_viewport ? 'Yes' : 'No';
		if ( ! $has_viewport ) {
			$score -= 20;
			$issues[] = array(
				'title'       => 'Not optimized for Mobile users',
				'description' => 'Your site is missing a viewport tag, which makes it look poor on smartphones.',
				'solution'    => 'Add <meta name="viewport" content="width=device-width, initial-scale=1"> to your site header.',
				'severity'    => 'high'
			);
		}

		return array( 'score' => max( 0, $score ), 'details' => $details, 'issues' => $issues, 'recs' => $recs );
	}

	private static function check_content( $html ) {
		$score  = 100;
		$issues = array();
		$recs   = array();
		$details = array();

		// Title tag
		if ( preg_match( '/<title>(.*?)<\/title>/is', $html, $matches ) ) {
			$title = trim( $matches[1] );
			$len   = strlen( $title );
			if ( $len < 30 ) {
				$score -= 10;
				$issues[] = array(
					'title'       => 'Page Title is too short',
					'description' => "Your title is only $len characters: \"$title\".",
					'solution'    => 'Rewrite your title to be between 50-60 characters to better describe your content to AI search engines.',
					'severity'    => 'low'
				);
			} elseif ( $len > 70 ) {
				$score -= 5;
				$issues[] = array(
					'title'       => 'Page Title is too long',
					'description' => "Your title is too long ($len chars): \"$title\".",
					'solution'    => 'Shorten your title to under 70 characters so it doesn\'t get cut off in search results.',
					'severity'    => 'low'
				);
			} else {
				$recs[] = array('title' => 'Optimized Title Tag', 'fix' => "Your page title is a good length ($len chars).");
			}
		} else {
			$score -= 25;
			$issues[] = array(
				'title'       => 'Missing Page Title',
				'description' => 'Your page has no title tag, making it impossible for AI search engines to understand the topic.',
				'solution'    => 'Add a descriptive <title> tag within the <head> section of your website.',
				'severity'    => 'high'
			);
		}

		// Meta Description
		if ( preg_match( '/<meta[^>]+name=["\']description["\'][^>]+content=["\'](.*?)["\']/i', $html, $matches ) ) {
			$desc = trim( $matches[1] );
			$len  = strlen( $desc );
			if ( $len < 70 ) {
				$score -= 10;
				$issues[] = array(
					'title'       => 'Description is too short',
					'description' => "Your description is too brief ($len chars): \"$desc\".",
					'solution'    => 'Expand your meta description to 120-160 characters to provide a better summary for AI answers.',
					'severity'    => 'medium'
				);
			} elseif ( $len > 165 ) {
				$score -= 5;
				$issues[] = array(
					'title'       => 'Description is too long',
					'description' => "Your description is too long ($len chars): \"$desc\".",
					'solution'    => 'Shorten your description to under 165 characters to ensure it displays correctly.',
					'severity'    => 'low'
				);
			} else {
				$recs[] = array('title' => 'Meta Description Present', 'fix' => 'Great! Your page has a well-optimized meta description.');
			}
		} else {
			$score -= 20;
			$issues[] = array(
				'title'       => 'Missing Meta Description',
				'description' => 'No meta description found. Description helps AI search engines summarize your content.',
				'solution'    => 'Add a <meta name="description" content="..."> tag to your site header with a 150-character summary.',
				'severity'    => 'high'
			);
		}

		// First Paragraph Analysis
		if ( preg_match( '/<p[^>]*>(.*?)<\/p>/is', $html, $matches ) ) {
			$para = trim( strip_tags( $matches[1] ) );
			$len  = strlen( $para );
			$details['First Paragraph'] = $len . ' chars';
			
			if ( $len < 100 ) {
				$score -= 10;
				$issues[] = array(
					'title'       => 'Introduction is too thin',
					'description' => "Your first paragraph is only $len characters. AI engines rely on the opening text for context.",
					'solution'    => 'Expand your introductory paragraph to at least 150-200 characters of high-quality, relevant information.',
					'severity'    => 'medium'
				);
			} else {
				$recs[] = array('title' => 'Strong Intro Paragraph', 'fix' => 'Your introductory paragraph has good substance (' . $len . ' chars).');
			}
		} else {
			$details['First Paragraph'] = 'Not detected';
			$score -= 15;
			$issues[] = array(
				'title'       => 'No Paragraph Content Detected',
				'description' => 'We couldn\'t find any standard <p> tags with content on your page.',
				'solution'    => 'Ensure your main content is wrapped in standard HTML <p> tags for better accessibility and AI parsing.',
				'severity'    => 'high'
			);
		}

		return array( 'score' => max( 0, $score ), 'issues' => $issues, 'recs' => $recs, 'details' => $details );
	}

	private static function check_structure( $html ) {
		$score  = 100;
		$issues = array();
		$recs   = array();
		$details = array();

		// H1 Tag
		$h1_count = preg_match_all( '/<h1/i', $html );
		$details['H1 Tags'] = $h1_count;
		if ( $h1_count === 0 ) {
			$score -= 25;
			$issues[] = array(
				'title'       => 'Missing Primary Heading (H1)',
				'description' => 'Your page is missing an H1 tag, which is the most important heading for SEO.',
				'solution'    => 'Identify the main topic of your page and wrap it in <h1> tags at the top of your content.',
				'severity'    => 'high'
			);
		} elseif ( $h1_count > 1 ) {
			preg_match_all( '/<h1[^>]*>(.*?)<\/h1>/is', $html, $h1_matches );
			$h1_list = implode( ', ', array_map( 'trim', array_map( 'strip_tags', $h1_matches[1] ) ) );
			$score -= 10;
			$issues[] = array(
				'title'       => 'Too many H1 tags found',
				'description' => "Found $h1_count H1 tags: \"$h1_list\".",
				'solution'    => 'Use only one H1 tag per page for the main title. Change the others to H2 or H3 tags.',
				'severity'    => 'medium'
			);
		} else {
			if ( preg_match( '/<h1[^>]*>(.*?)<\/h1>/is', $html, $h1_match ) ) {
				$recs[] = array('title' => 'Primary H1 Found', 'fix' => 'Your primary heading is correctly set: "' . trim(strip_tags($h1_match[1])) . '"');
			}
		}

		// H2 Tags
		$h2_count = preg_match_all( '/<h2/i', $html );
		$details['H2 Tags'] = $h2_count;
		if ( $h2_count < 2 ) {
			$score -= 10;
			$issues[] = array(
				'title'       => 'Few Subheadings (H2)',
				'description' => 'Your page uses very few H2 tags. Subheadings help organize content for Answer Engines.',
				'solution'    => 'Break up your content into logical sections and use H2 tags for each section title.',
				'severity'    => 'medium'
			);
		} else {
			$recs[] = array('title' => 'Good Heading Structure', 'fix' => "Found $h2_count H2 tags helping to structure your content.");
		}

		// FAQ Section Detection
		$has_faq_markup = ( stripos( $html, 'FAQPage' ) !== false );
		$has_faq_header = ( preg_match( '/<(h2|h3)[^>]*>(.*?)faq(s|.*?question|.*?answer).*?<\/(h2|h3)>/i', $html ) );
		$details['FAQ Section'] = ( $has_faq_markup || $has_faq_header ) ? '✅ Found' : '❌ Missing';
		
		if ( ! ( $has_faq_markup || $has_faq_header ) ) {
			$score -= 15;
			$issues[] = array(
				'title'       => 'No FAQ Section Detected',
				'description' => 'FAQ sections are highly effective for capturing "featured snippets" and appearing in AI results.',
				'solution'    => 'Add a section with common questions and answers related to your content, and use FAQ schema markup if possible.',
				'severity'    => 'medium'
			);
		} else {
			$recs[] = array('title' => 'FAQ Section Detected', 'fix' => 'Your page includes an FAQ section, which is excellent for AEO.');
		}

		// Schema Markup (Basic check)
		$has_schema = ( stripos( $html, 'ld+json' ) !== false || stripos( $html, 'itemtype' ) !== false );
		$details['Schema Markup'] = $has_schema ? '✅ Found' : '❌ Missing';
		if ( ! $has_schema ) {
			$score -= 20;
			$issues[] = array(
				'title'       => 'No Structured Data (Schema) detected',
				'description' => 'Schema markup helps AI search engines like Perplexity and ChatGPT understand your content better.',
				'solution'    => 'Use an SEO plugin to add Schema.org markup (like Article or WebSite) to your pages.',
				'severity'    => 'high'
			);
		} else {
			$recs[] = array('title' => 'Schema Detected', 'fix' => 'Your page uses structured data, which helps AEs understand context.');
		}

		return array( 'score' => max( 0, $score ), 'issues' => $issues, 'recs' => $recs, 'details' => $details );
	}

	private static function get_error_response( $url, $message ) {
		return array(
			'url'               => $url,
			'score'             => 0,
			'status'            => 'Error',
			'breakdown'         => array( 'content' => 0, 'technical' => 0, 'structure' => 0 ),
			'technical_details' => array( 'Error' => $message ),
			'issues'            => array( array( 'title' => 'Scan Failed', 'description' => "We couldn't analyze $url: $message", 'severity' => 'high' ) ),
			'recommendations'   => array(),
		);
	}
}
