<?php
/**
 * API Client
 * Future-ready API client with fallback to mock responses
 *
 * @package AEOWPPlugin
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AEOWPPlugin_API_Client {

	/**
	 * API Base URL
	 */
	const API_BASE_URL = 'https://aeowpplugin.com/api';

	/**
	 * Validate license via API
	 *
	 * @param string $license_key License key
	 * @return array|WP_Error Response or error
	 */
	public static function validate_license( $license_key ) {
		$endpoint = self::API_BASE_URL . '/license/validate';

		$response = wp_remote_post( $endpoint, array(
			'timeout' => 15,
			'headers' => array(
				'Content-Type' => 'application/json',
			),
			'body'    => wp_json_encode( array(
				'licenseKey' => $license_key,
			) ),
		) );

		// If request fails, return error (will fallback to mock)
		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		// If API is not available (404, 500, etc), return error to trigger fallback
		if ( $code !== 200 ) {
			return new WP_Error( 'api_unavailable', 'API endpoint not available' );
		}

		// Return API response
		return $data;
	}

	/**
	 * Perform scan via API
	 *
	 * @param string $url URL to scan
	 * @param string $license_key License key
	 * @return array|WP_Error Response or error
	 */
	public static function perform_scan( $url, $license_key ) {
		$endpoint = self::API_BASE_URL . '/scan';

		$response = wp_remote_post( $endpoint, array(
			'timeout' => 30,
			'headers' => array(
				'Content-Type' => 'application/json',
			),
			'body'    => wp_json_encode( array(
				'url'        => $url,
				'licenseKey' => $license_key,
			) ),
		) );

		// If request fails, return error (will fallback to mock)
		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		// If API returns error, pass it through
		if ( $code !== 200 ) {
			$message = isset( $data['message'] ) ? $data['message'] : 'API request failed';
			return new WP_Error( 'api_error', $message );
		}

		// Return scan results
		return $data;
	}

	/**
	 * Mock scan response (fallback when API is unavailable)
	 *
	 * @param string $url URL to scan
	 * @return array Mock scan results
	 */
	public static function mock_scan_response( $url ) {
		// Replace random generation with real local analysis
		$analysis = AEOWPPlugin_Analyzer::run_analysis( $url );
		
		return array(
			'success' => true,
			'data'    => $analysis,
		);
	}

	/**
	 * Check API health
	 *
	 * @return bool API is available
	 */
	public static function is_api_available() {
		$response = wp_remote_get( self::API_BASE_URL . '/health', array(
			'timeout' => 5,
		) );

		if ( is_wp_error( $response ) ) {
			return false;
		}

		$code = wp_remote_retrieve_response_code( $response );
		return $code === 200;
	}
}
