<?php
/**
 * License Manager
 * Handles license validation, activation, and deactivation
 *
 * @package AEOWPPlugin
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AEOWPPlugin_License_Manager {

	/**
	 * Plan configurations
	 */
	const PLANS = array(
		'free' => array(
			'name'   => 'Free',
			'limit'  => 5,
			'period' => 'month',
			'price'  => 0,
		),
		'weekly' => array(
			'name'   => 'Weekly',
			'limit'  => 25,
			'period' => 'week',
			'price'  => 9,
		),
		'pro' => array(
			'name'   => 'Pro',
			'limit'  => 150,
			'period' => 'month',
			'price'  => 29,
		),
		'enterprise' => array(
			'name'   => 'Enterprise',
			'limit'  => 500,
			'period' => 'month',
			'price'  => 99,
		),
	);

	/**
	 * Validate license key
	 *
	 * @param string $license_key License key to validate
	 * @return array|WP_Error Validation response or error
	 */
	public static function validate_license( $license_key ) {
		// First try real API
		$api_response = AEOWPPlugin_API_Client::validate_license( $license_key );

		// If API is available and returns valid response, use it
		if ( ! is_wp_error( $api_response ) && isset( $api_response['valid'] ) ) {
			return $api_response;
		}

		// Fallback to mock validation
		return self::mock_validate_license( $license_key );
	}

	/**
	 * Mock license validation (for testing before API is ready)
	 *
	 * @param string $license_key License key
	 * @return array Validation response
	 */
	private static function mock_validate_license( $license_key ) {
		$license_key = trim( $license_key );

		// Empty or invalid
		if ( empty( $license_key ) ) {
			return array(
				'valid'      => false,
				'plan'       => 'free',
				'status'     => 'inactive',
				'message'    => 'No license key provided',
				'expires_at' => null,
				'quota'      => array(
					'period'    => 'month',
					'limit'     => 5,
					'used'      => 0,
					'remaining' => 5,
				),
			);
		}

		// Determine plan from prefix
		$plan = 'free';
		if ( strpos( $license_key, 'FREE-' ) === 0 ) {
			$plan = 'free';
		} elseif ( strpos( $license_key, 'WEEKLY-' ) === 0 ) {
			$plan = 'weekly';
		} elseif ( strpos( $license_key, 'PRO-' ) === 0 ) {
			$plan = 'pro';
		} elseif ( strpos( $license_key, 'ENT-' ) === 0 ) {
			$plan = 'enterprise';
		} else {
			// Invalid format
			return array(
				'valid'      => false,
				'plan'       => 'free',
				'status'     => 'invalid',
				'message'    => 'Invalid license key format',
				'expires_at' => null,
				'quota'      => array(
					'period'    => 'month',
					'limit'     => 5,
					'used'      => 0,
					'remaining' => 5,
				),
			);
		}

		$plan_config = self::PLANS[ $plan ];
		$quota_data  = AEOWPPlugin_Quota_Manager::get_quota_status();

		return array(
			'valid'      => true,
			'plan'       => $plan,
			'status'     => 'active',
			'message'    => 'License activated successfully',
			'expires_at' => date( 'Y-m-d', strtotime( '+1 year' ) ),
			'quota'      => array(
				'period'    => $plan_config['period'],
				'limit'     => $plan_config['limit'],
				'used'      => $quota_data['used'],
				'remaining' => max( 0, $plan_config['limit'] - $quota_data['used'] ),
			),
		);
	}

	/**
	 * Activate license
	 *
	 * @param string $license_key License key
	 * @return array|WP_Error Response
	 */
	public static function activate_license( $license_key ) {
		$validation = self::validate_license( $license_key );

		if ( isset( $validation['valid'] ) && $validation['valid'] ) {
			// Store license data
			update_option( 'aeowpplugin_license_key', $license_key );
			update_option( 'aeowpplugin_license_data', $validation );
			update_option( 'aeowpplugin_license_activated_at', current_time( 'timestamp' ) );

			// Update quota limits based on plan
			AEOWPPlugin_Quota_Manager::set_plan_limits( $validation['plan'] );

			return $validation;
		}

		return new WP_Error( 'invalid_license', $validation['message'] ?? 'Invalid license key' );
	}

	/**
	 * Deactivate license
	 *
	 * @return bool Success
	 */
	public static function deactivate_license() {
		delete_option( 'aeowpplugin_license_key' );
		delete_option( 'aeowpplugin_license_data' );
		delete_option( 'aeowpplugin_license_activated_at' );

		// Reset to free plan
		AEOWPPlugin_Quota_Manager::set_plan_limits( 'free' );

		return true;
	}

	/**
	 * Get current license data
	 *
	 * @return array License data
	 */
	public static function get_license_data() {
		$license_data = get_option( 'aeowpplugin_license_data', array() );
		
		// Always fetch live quota status to avoid stale data
		$quota = array(
			'period'    => 'month',
			'limit'     => 5,
			'used'      => 0,
			'remaining' => 5,
		);

		if ( class_exists( 'AEOWPPlugin_Quota_Manager' ) ) {
			try {
				$quota = AEOWPPlugin_Quota_Manager::get_quota_status();
			} catch ( Exception $e ) {
				// Fallback to defaults
			}
		}

		// If no license data, return free plan defaults
		if ( empty( $license_data ) ) {
			return array(
				'valid'      => false,
				'plan'       => 'free',
				'status'     => 'inactive',
				'message'    => 'No active license',
				'expires_at' => null,
				'quota'      => $quota,
			);
		}

		// Merge live quota into stored license data
		$license_data['quota'] = $quota;
		return $license_data;
	}

	/**
	 * Get current plan
	 *
	 * @return string Plan name
	 */
	public static function get_current_plan() {
		$license_data = self::get_license_data();
		return $license_data['plan'] ?? 'free';
	}

	/**
	 * Get plan configuration
	 *
	 * @param string $plan Plan name
	 * @return array Plan config
	 */
	public static function get_plan_config( $plan ) {
		return self::PLANS[ $plan ] ?? self::PLANS['free'];
	}

	/**
	 * Check if license is active
	 *
	 * @return bool
	 */
	public static function is_license_active() {
		$license_data = self::get_license_data();
		return isset( $license_data['valid'] ) && $license_data['valid'] && $license_data['status'] === 'active';
	}
}
