<?php
/**
 * Quota Manager
 * Handles usage tracking and quota enforcement
 *
 * @package AEOWPPlugin
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AEOWPPlugin_Quota_Manager {

	/**
	 * Check if user can perform a scan
	 *
	 * @return array Status with 'allowed' boolean and 'message'
	 */
	public static function can_scan() {
		$quota = self::get_quota_status();

		if ( $quota['remaining'] <= 0 ) {
			return array(
				'allowed' => false,
				'message' => sprintf(
					'You have reached your %s limit of %d scans. Upgrade anytime to continue.',
					$quota['period'],
					$quota['limit']
				),
				'quota'   => $quota,
			);
		}

		return array(
			'allowed' => true,
			'message' => 'Scan allowed',
			'quota'   => $quota,
		);
	}

	/**
	 * Increment scan usage
	 *
	 * @return bool Success
	 */
	public static function increment_usage() {
		$quota_data = get_option( 'aeowpplugin_quota_data', array(
			'used'       => 0,
			'last_reset' => current_time( 'timestamp' ),
		) );

		$quota_data['used']++;
		update_option( 'aeowpplugin_quota_data', $quota_data );

		return true;
	}

	/**
	 * Get current quota status
	 *
	 * @return array Quota status
	 */
	public static function get_quota_status() {
		// Get current plan data directly to avoid circular dependency
		$license_data = get_option( 'aeowpplugin_license_data', array() );
		$plan         = $license_data['plan'] ?? 'free';

		if ( ! class_exists( 'AEOWPPlugin_License_Manager' ) ) {
			$plan_config = array(
				'period' => 'month',
				'limit'  => 5,
			);
		} else {
			$plan_config = AEOWPPlugin_License_Manager::get_plan_config( $plan );
		}

		// Get usage data
		$quota_data = get_option( 'aeowpplugin_quota_data', array(
			'used'       => 0,
			'last_reset' => current_time( 'timestamp' ),
		) );

		// Check if we need to reset quota
		$should_reset = self::should_reset_quota( $quota_data['last_reset'], $plan_config['period'] );

		if ( $should_reset ) {
			$quota_data = array(
				'used'       => 0,
				'last_reset' => current_time( 'timestamp' ),
			);
			update_option( 'aeowpplugin_quota_data', $quota_data );
		}

		$used      = $quota_data['used'];
		$limit     = $plan_config['limit'];
		$remaining = max( 0, $limit - $used );

		return array(
			'period'     => $plan_config['period'],
			'limit'      => $limit,
			'used'       => $used,
			'remaining'  => $remaining,
			'last_reset' => $quota_data['last_reset'],
			'next_reset' => self::get_next_reset_date( $quota_data['last_reset'], $plan_config['period'] ),
		);
	}

	/**
	 * Check if quota should be reset
	 *
	 * @param int    $last_reset Last reset timestamp
	 * @param string $period Period type (week/month)
	 * @return bool Should reset
	 */
	private static function should_reset_quota( $last_reset, $period ) {
		$now = current_time( 'timestamp' );

		if ( $period === 'week' ) {
			// Reset every 7 days
			return ( $now - $last_reset ) >= ( 7 * DAY_IN_SECONDS );
		}

		// Monthly reset - check if we're in a new month
		$last_reset_month = date( 'Y-m', $last_reset );
		$current_month    = date( 'Y-m', $now );

		return $last_reset_month !== $current_month;
	}

	/**
	 * Get next reset date
	 *
	 * @param int    $last_reset Last reset timestamp
	 * @param string $period Period type
	 * @return int Next reset timestamp
	 */
	private static function get_next_reset_date( $last_reset, $period ) {
		if ( $period === 'week' ) {
			return $last_reset + ( 7 * DAY_IN_SECONDS );
		}

		// Monthly - first day of next month
		$last_reset_date = date( 'Y-m-01', $last_reset );
		return strtotime( '+1 month', strtotime( $last_reset_date ) );
	}

	/**
	 * Set plan limits (called when license is activated)
	 *
	 * @param string $plan Plan name
	 * @return bool Success
	 */
	public static function set_plan_limits( $plan ) {
		// Only proceed if License Manager is loaded
		if ( class_exists( 'AEOWPPlugin_License_Manager' ) ) {
			$plan_config = AEOWPPlugin_License_Manager::get_plan_config( $plan );
		}

		// Reset quota when changing plans
		update_option( 'aeowpplugin_quota_data', array(
			'used'       => 0,
			'last_reset' => current_time( 'timestamp' ),
		) );

		return true;
	}

	/**
	 * Reset quota manually (for testing)
	 *
	 * @return bool Success
	 */
	public static function reset_quota() {
		update_option( 'aeowpplugin_quota_data', array(
			'used'       => 0,
			'last_reset' => current_time( 'timestamp' ),
		) );

		return true;
	}

	/**
	 * Get usage percentage
	 *
	 * @return float Percentage (0-100)
	 */
	public static function get_usage_percentage() {
		$quota = self::get_quota_status();

		if ( $quota['limit'] <= 0 ) {
			return 0;
		}

		return min( 100, ( $quota['used'] / $quota['limit'] ) * 100 );
	}
}
