<?php
/**
 * AEOWPPlugin Updater
 * Handles premium plugin updates via custom API
 *
 * @package AEOWPPlugin
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AEOWPPlugin_Updater {

	private $version;
	private $api_url = 'https://aeowpplugin.com/api/plugin/update';
	private $plugin_slug;
	private $license_key;

	public function __construct( $version, $plugin_slug ) {
		$this->version     = $version;
		$this->plugin_slug = $plugin_slug;
		$this->license_key = get_option( 'aeowpplugin_license_key', '' );

		// Check for updates
		add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'check_update' ) );

		// Provide plugin info (changelog)
		add_filter( 'plugins_api', array( $this, 'plugin_popup' ), 10, 3 );
	}

	/**
	 * Check for updates
	 */
	public function check_update( $transient ) {
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$remote = $this->request_info();

		if ( $remote && version_compare( $this->version, $remote->version, '<' ) ) {
			$res = new stdClass();
			$res->slug         = $this->plugin_slug;
			$res->plugin       = $this->plugin_slug . '/' . $this->plugin_slug . '.php';
			$res->new_version  = $remote->version;
			$res->tested       = $remote->tested;
			$res->package      = $remote->download_url;
			$res->icons        = (array) $remote->icons;
			$res->banners      = (array) $remote->banners;
			$res->sections     = (array) $remote->sections;

			$transient->response[ $res->plugin ] = $res;
		}

		return $transient;
	}

	/**
	 * Provide plugin details for the popup
	 */
	public function plugin_popup( $res, $action, $args ) {
		if ( 'plugin_information' !== $action ) {
			return $res;
		}

		if ( $this->plugin_slug !== $args->slug ) {
			return $res;
		}

		$remote = $this->request_info();

		if ( ! $remote ) {
			return $res;
		}

		$res = new stdClass();
		$res->name           = $remote->name;
		$res->slug           = $this->plugin_slug;
		$res->version        = $remote->version;
		$res->tested         = $remote->tested;
		$res->requires       = $remote->requires;
		$res->author         = $remote->author;
		$res->author_profile = $remote->author_profile;
		$res->download_link  = $remote->download_url;
		$res->trunk          = $remote->download_url;
		$res->requires_php   = $remote->requires_php;
		$res->last_updated   = $remote->last_updated;

		$res->sections = array(
			'description'  => $remote->sections->description,
			'installation' => $remote->sections->installation,
			'changelog'    => $remote->sections->changelog,
		);

		if ( ! empty( $remote->banners ) ) {
			$res->banners = (array) $remote->banners;
		}

		return $res;
	}

	/**
	 * Request update info from API
	 */
	private function request_info() {
		$cache_key = 'aeowpplugin_update_check';
		$remote    = get_transient( $cache_key );

		if ( false !== $remote ) {
			return $remote;
		}

		$response = wp_remote_post( $this->api_url, array(
			'timeout' => 15,
			'headers' => array(
				'Content-Type'      => 'application/json',
				'X-Plugin-License'  => $this->license_key,
			),
			'body'    => wp_json_encode( array(
				'slug'    => $this->plugin_slug,
				'version' => $this->version,
			) ),
		) );

		if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
			return false;
		}

		$remote = json_decode( wp_remote_retrieve_body( $response ) );
		
		if ( ! $remote || ! isset( $remote->version ) ) {
			return false;
		}

		set_transient( $cache_key, $remote, 12 * HOUR_IN_SECONDS );

		return $remote;
	}
}
