<?php
/**
 * Gutenberg Editor Integration for AEO WP Plugin
 * Adds an AEO sidebar panel in the block editor
 *
 * @package AEOWPPlugin
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Register Gutenberg sidebar
 */
add_action( 'enqueue_block_editor_assets', 'aeowpplugin_enqueue_gutenberg_assets' );

function aeowpplugin_enqueue_gutenberg_assets() {
	// Enqueue script
	wp_enqueue_script(
		'aeowpplugin-gutenberg',
		AEOWPPLUGIN_URL . 'assets/js/gutenberg-sidebar.js',
		array( 'wp-plugins', 'wp-edit-post', 'wp-element', 'wp-components', 'wp-api-fetch', 'chart-js' ),
		AEOWPPLUGIN_VERSION,
		true
	);

	// Pass plugin URL and nonce to script
	wp_localize_script(
		'aeowpplugin-gutenberg',
		'aeowppluginGutenberg',
		array(
			'pluginUrl'     => AEOWPPLUGIN_URL,
			'nonce'         => wp_create_nonce( 'aeowpplugin_nonce' ),
			'restUrl'       => rest_url(),
			'postId'        => get_the_ID(),
			'postPermalink' => get_permalink(),
		)
	);

	// Enqueue styles
	wp_enqueue_style(
		'aeowpplugin-gutenberg',
		AEOWPPLUGIN_URL . 'assets/css/gutenberg-sidebar.css',
		array(),
		AEOWPPLUGIN_VERSION
	);
}

/**
 * Add REST endpoint for post scanning
 */
add_action( 'rest_api_init', 'aeowpplugin_register_gutenberg_routes' );

function aeowpplugin_register_gutenberg_routes() {
	register_rest_route(
		'aeowpplugin/v1',
		'/gutenberg-scan',
		array(
			'methods'             => 'POST',
			'callback'            => 'aeowpplugin_handle_gutenberg_scan',
			'permission_callback' => function() {
				return current_user_can( 'edit_posts' );
			},
		)
	);
}

/**
 * Handle scan from Gutenberg sidebar
 *
 * @param WP_REST_Request $request REST request object.
 * @return WP_REST_Response
 */
function aeowpplugin_handle_gutenberg_scan( $request ) {
	// Verify nonce
	$nonce = $request->get_header( 'X-WP-Nonce' );
	if ( ! wp_verify_nonce( $nonce, 'aeowpplugin_nonce' ) ) {
		return new WP_REST_Response(
			array( 'error' => __( 'Security verification failed.', 'aeowpplugin' ) ),
			403
		);
	}

	$params = $request->get_json_params();
	$post_url = isset( $params['url'] ) ? esc_url_raw( $params['url'] ) : '';

	if ( empty( $post_url ) ) {
		return new WP_REST_Response(
			array( 'error' => __( 'URL is required.', 'aeowpplugin' ) ),
			400
		);
	}

	// Get API key
	$api_key = get_option( 'aeowpplugin_api_key' );
	if ( empty( $api_key ) ) {
		return new WP_REST_Response(
			array( 'error' => __( 'API key is not configured.', 'aeowpplugin' ) ),
			400
		);
	}

	// Use the same API handler as dashboard
	require_once AEOWPPLUGIN_PATH . 'includes/api-handler.php';
	
	// Call external API (same as dashboard)
	$response = aeowpplugin_call_api( $post_url, $api_key );

	if ( is_wp_error( $response ) ) {
		return new WP_REST_Response(
			array( 'error' => $response->get_error_message() ),
			500
		);
	}

	return new WP_REST_Response( $response, 200 );
}

/**
 * Call external API for scanning
 *
 * @param string $url Target URL.
 * @param string $api_key API key.
 * @return array|WP_Error
 */
function aeowpplugin_call_api( $url, $api_key ) {
	$response = wp_remote_post(
		'https://api.aeowpplugin.com/scan',
		array(
			'headers' => array(
				'Authorization' => 'Bearer ' . sanitize_text_field( $api_key ),
				'Content-Type'  => 'application/json',
			),
			'body'    => wp_json_encode( array( 'url' => $url ) ),
			'timeout' => 30,
		)
	);

	if ( is_wp_error( $response ) ) {
		return $response;
	}

	$code = wp_remote_retrieve_response_code( $response );

	if ( 429 === $code ) {
		return new WP_Error(
			'rate_limited',
			__( 'API rate limit exceeded. Please upgrade your plan.', 'aeowpplugin' )
		);
	}

	if ( 401 === $code || 403 === $code ) {
		return new WP_Error(
			'auth_failed',
			__( 'Invalid or expired API key.', 'aeowpplugin' )
		);
	}

	if ( 200 !== $code ) {
		return new WP_Error(
			'api_error',
			__( 'Failed to scan the URL. Please try again.', 'aeowpplugin' )
		);
	}

	$body = wp_remote_retrieve_body( $response );
	$data = json_decode( $body, true );

	// Mock response if empty
	if ( empty( $data ) ) {
		$data = array(
			'score'             => rand( 40, 95 ),
			'status'            => array_rand( array_flip( array( 'Poor', 'Average', 'Good', 'Excellent' ) ) ),
			'issues'            => array(
				array(
					'title'       => 'Missing Structured Data',
					'description' => 'Your post is missing JSON-LD schema.',
					'severity'    => 'high',
				),
			),
			'recommendations'   => array(
				array(
					'title' => 'Add FAQ Schema',
					'fix'   => 'Include FAQ structured data for better AI understanding.',
				),
			),
		);
	}

	return $data;
}
