<?php
/**
 * Helper Functions
 *
 * @package AEOWPPlugin
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Get formatted plan name
 *
 * @param string $plan Plan slug
 * @return string Formatted plan name
 */
function aeowpplugin_get_plan_name( $plan ) {
	$plans = array(
		'free'       => 'Free',
		'weekly'     => 'Weekly',
		'pro'        => 'Pro',
		'enterprise' => 'Enterprise',
	);

	return $plans[ $plan ] ?? 'Free';
}

/**
 * Get plan badge class
 *
 * @param string $plan Plan slug
 * @return string CSS class
 */
function aeowpplugin_get_plan_badge_class( $plan ) {
	$classes = array(
		'free'       => 'badge-free',
		'weekly'     => 'badge-weekly',
		'pro'        => 'badge-pro',
		'enterprise' => 'badge-enterprise',
	);

	return $classes[ $plan ] ?? 'badge-free';
}

/**
 * Format date for display
 *
 * @param string $date Date string
 * @return string Formatted date
 */
function aeowpplugin_format_date( $date ) {
	if ( empty( $date ) ) {
		return 'N/A';
	}

	return date_i18n( get_option( 'date_format' ), strtotime( $date ) );
}

/**
 * Get upgrade URL
 *
 * @return string Upgrade URL
 */
function aeowpplugin_get_upgrade_url() {
	return 'https://aeowpplugin.com/pricing';
}

/**
 * Check if user has premium plan
 *
 * @return bool
 */
function aeowpplugin_is_premium() {
	$plan = AEOWPPlugin_License_Manager::get_current_plan();
	return in_array( $plan, array( 'weekly', 'pro', 'enterprise' ), true );
}

/**
 * Get status badge HTML
 *
 * @param string $status Status
 * @return string HTML
 */
function aeowpplugin_get_status_badge( $status ) {
	$classes = array(
		'active'   => 'badge-active',
		'inactive' => 'badge-inactive',
		'expired'  => 'badge-expired',
		'invalid'  => 'badge-invalid',
	);

	$class = $classes[ $status ] ?? 'badge-inactive';

	return sprintf(
		'<span class="aeowpplugin-status-badge %s">%s</span>',
		esc_attr( $class ),
		esc_html( ucfirst( $status ) )
	);
}
/**
 * Get current update status
 *
 * @return array Status data
 */
function aeowpplugin_get_update_status() {
	$remote = get_transient( 'aeowpplugin_update_check' );
	$version = AEOWPPLUGIN_VERSION;

	if ( ! $remote ) {
		return array(
			'version' => $version,
			'status'  => 'Up to date',
			'msg'     => __( 'You are using the latest version.', 'aeowpplugin' ),
		);
	}

	$is_uptodate = version_compare( $version, $remote->version, '>=' );

	return array(
		'version' => $version,
		'latest'  => $remote->version,
		'status'  => $is_uptodate ? 'Up to date' : 'Update Available',
		'msg'     => $is_uptodate ? __( 'You are using the latest version.', 'aeowpplugin' ) : sprintf( __( 'Version %s is available.', 'aeowpplugin' ), $remote->version ),
	);
}
