<?php
/**
 * Rate Limiting for AEO WP Plugin
 *
 * @package AEOWPPlugin
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * AEOWPPlugin_Rate_Limit class
 */
class AEOWPPlugin_Rate_Limit {

	const SCANS_PER_DAY = 10; // Free tier limit

	/**
	 * Check if scan is allowed
	 *
	 * @return array
	 */
	public static function check_rate_limit() {
		$transient_key = 'aeowpplugin_daily_scans_' . gmdate( 'Y-m-d' );
		$scan_count    = get_transient( $transient_key );

		if ( false === $scan_count ) {
			$scan_count = 0;
		}

		$allowed   = $scan_count < self::SCANS_PER_DAY;
		$retry_after = $allowed ? 0 : 86400; // 24 hours in seconds

		return array(
			'allowed'      => $allowed,
			'used'         => $scan_count,
			'limit'        => self::SCANS_PER_DAY,
			'retry_after'  => $retry_after,
		);
	}

	/**
	 * Record a scan in rate limit counter
	 */
	public static function record_scan() {
		$transient_key = 'aeowpplugin_daily_scans_' . gmdate( 'Y-m-d' );
		$scan_count    = get_transient( $transient_key );

		if ( false === $scan_count ) {
			$scan_count = 0;
		}

		$scan_count++;
		set_transient( $transient_key, $scan_count, 86400 ); // 24 hours
	}

	/**
	 * Get rate limit info
	 *
	 * @return array
	 */
	public static function get_rate_limit_info() {
		$transient_key = 'aeowpplugin_daily_scans_' . gmdate( 'Y-m-d' );
		$scan_count    = get_transient( $transient_key );

		if ( false === $scan_count ) {
			$scan_count = 0;
		}

		return array(
			'used'      => $scan_count,
			'limit'     => self::SCANS_PER_DAY,
			'remaining' => max( 0, self::SCANS_PER_DAY - $scan_count ),
		);
	}
}
